#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.Tools;
using NinjaTrader.Data;
#endregion


	public enum PT_TextLocation
	{
		Off,
		Extreme,
		Inside
	};


	public enum PT_AlertCondition
	{
		CrossAbove,
		CrossBelow,
		Equal,
		Any
	};
	
	public enum PT_AlertRearmOn
	{
		Never,
		OnTimer,
		BarClose,
		ConditionReversed
	};
	
namespace NinjaTrader.NinjaScript.DrawingTools
{

	// note: when using a type converter attribute with dynamically loaded assemblies such as NinjaTrader custom,
	// you must pass typeconverter a string parameter. passing a type will fail to resolve
	/// <summary>
	/// Represents an interface that exposes information regarding a Fibonacci Circle IDrawingTool.
	/// </summary>
	[EditorBrowsable(EditorBrowsableState.Always)]
	[TypeConverter("NinjaTrader.NinjaScript.DrawingTools.PT_FibonacciCircleTimeTypeConverter")]
	public class PT_FibonacciCircle : PT_FibonacciFan
	{
		[Display(ResourceType=typeof(Custom.Resource), Name = "ShowText", GroupName = "NinjaScriptGeneral")]
		public bool IsTextDisplayed { get; set; }

		[Display(ResourceType=typeof(Custom.Resource), Name = "DrawEllipse", GroupName = "NinjaScriptGeneral")]
		public bool IsDrawEllipse { get; set; }
		
		public override void OnCalculateMinMax()
		{
			MinValue = double.MaxValue;
			MaxValue = double.MinValue;

			if (!IsVisible)
				return;

			foreach (ChartAnchor anchor in Anchors)
			{ 
				MinValue = Math.Min(MinValue, anchor.Price);
				MaxValue = Math.Max(MaxValue, anchor.Price);
			}
		}
	
		private void DrawPriceLevelText(float textX, float textY, PT_PriceLevel priceLevel, double yVal, ChartControl chartControl)
		{
			if (!IsTextDisplayed)
				return;
			
			SimpleFont						wpfFont		= chartControl.Properties.LabelFont ?? new SimpleFont();
			SharpDX.DirectWrite.TextFormat	textFormat	= wpfFont.ToDirectWriteTextFormat();
			textFormat.TextAlignment					= SharpDX.DirectWrite.TextAlignment.Leading;
			textFormat.WordWrapping						= SharpDX.DirectWrite.WordWrapping.NoWrap;
			string							str			= GetPriceString(yVal, priceLevel);

			SharpDX.DirectWrite.TextLayout textLayout  = new SharpDX.DirectWrite.TextLayout(Core.Globals.DirectWriteFactory, str, textFormat, 250, textFormat.FontSize);
			RenderTarget.DrawTextLayout(new SharpDX.Vector2(textX, textY), textLayout, priceLevel.Stroke.BrushDX, SharpDX.Direct2D1.DrawTextOptions.NoSnap);

			textFormat.Dispose();
			textLayout.Dispose();
		}
	
		
		private string GetPriceString(double yVal, PT_PriceLevel priceLevel)
		{
			string							priceStr	= yVal.ToString(Core.Globals.GetTickFormatString(AttachedTo.Instrument.MasterInstrument.TickSize));
			string							str			= string.Format("{0} ({1})", (priceLevel.Value / 100).ToString("P", Core.Globals.GeneralOptions.CurrentCulture), priceStr);
			return str;
		}

		private string GetPercentString (PT_PriceLevel priceLevel)
		{
			string							str			= string.Format("{0}", (priceLevel.Value / 100).ToString("P", Core.Globals.GeneralOptions.CurrentCulture));
			return str;
		}
		
		public override IEnumerable<Condition> GetValidAlertConditions()
		{
			return new[] { Condition.CrossInside, Condition.CrossOutside };
		}

		public override bool IsAlertConditionTrue(AlertConditionItem conditionItem, Condition condition, ChartAlertValue[] values,
													ChartControl chartControl, ChartScale chartScale)
		{
			if (DrawingState == DrawingState.Building)
				return false;
			PT_PriceLevel priceLevel = conditionItem.Tag as PT_PriceLevel;
			if (priceLevel == null)
				return false;

			ChartPanel chartPanel	= chartControl.ChartPanels[PanelIndex];
			// get our data ellipse for given price level alert is on
			Point anchorStartPoint 	= StartAnchor.GetPoint(chartControl, chartPanel, chartScale);
			Point anchorEndPoint 	= EndAnchor.GetPoint(chartControl, chartPanel, chartScale);
			
			double indexRange = chartControl.GetSlotIndexByX(Convert.ToInt32(anchorEndPoint.X)) - chartControl.GetSlotIndexByX(Convert.ToInt32(anchorStartPoint.X));
			double ticksRange = (EndAnchor.Price - StartAnchor.Price)/chartControl.Instrument.MasterInstrument.TickSize;
			double radius = Math.Sqrt(Math.Pow(indexRange, 2) + Math.Pow(ticksRange, 2));
			double xRange = radius * chartControl.Properties.BarDistance ;
			double yRange = chartScale.GetPixelsForDistance(radius*chartControl.Instrument.MasterInstrument.TickSize);
			
		
			float levelFactor 	= (float)priceLevel.Value/100f;
			float xScale 		= (float)(levelFactor * xRange);
			float yScale 		= (float)(levelFactor * yRange);
			
			// NOTE: don't divide by two for the IsPointInsideEllipse check, this is the correct value we want already (center->r)
			float ellipseRadiusX = xScale;
			float ellipseRadiusY = yScale;

			Point centerPoint = StartAnchor.GetPoint(chartControl, chartPanel, chartScale); 
			
			
			Predicate<ChartAlertValue> predicate = v =>
			{
				bool dirUp = StartAnchor.Price < EndAnchor.Price;
				Point	barPoint = new Point(chartControl.GetXByTime(v.Time), chartScale.GetYByValue(v.Value));
				bool	isInside = MathHelper.IsPointInsideEllipse(centerPoint, barPoint, ellipseRadiusX, ellipseRadiusY);
				if (!IsDrawEllipse)
				{
					if (dirUp)
					{
						if (barPoint.Y > centerPoint.Y)
							isInside = false;
					}
					else
					{
						if (barPoint.Y < centerPoint.Y)
							isInside = false;						
					}							
				}
				return	condition == Condition.CrossInside ? isInside : !isInside;
			};
			return MathHelper.DidPredicateCross(values, predicate);
		}

		public override bool IsVisibleOnChart(ChartControl chartControl, ChartScale chartScale, DateTime firstTimeOnChart, DateTime lastTimeOnChart)
		{
			if (DrawingState == DrawingState.Building)
				return true;

			// find the biggest ellipse level and use that for visibility bounds
			double biggestPriceLevelValue	= PriceLevels.Max(pl => pl.Value);
			ChartPanel chartPanel			= chartControl.ChartPanels[PanelIndex];
			Point anchorStartPoint 			= StartAnchor.GetPoint(chartControl, chartPanel, chartScale);
			Point anchorEndPoint 			= EndAnchor.GetPoint(chartControl, chartPanel, chartScale);
			
			
			
			double indexRange = chartControl.GetSlotIndexByX(Convert.ToInt32(anchorEndPoint.X)) - chartControl.GetSlotIndexByX(Convert.ToInt32(anchorStartPoint.X));
			double ticksRange = (EndAnchor.Price - StartAnchor.Price)/chartControl.Instrument.MasterInstrument.TickSize;
			double radius = Math.Sqrt(Math.Pow(indexRange, 2) + Math.Pow(ticksRange, 2));
			double xRange = radius * chartControl.Properties.BarDistance ;
			double yRange = chartScale.GetPixelsForDistance(radius*chartControl.Instrument.MasterInstrument.TickSize);
			
			float levelFactor 				= (float)biggestPriceLevelValue/100f;
		
			float xScale 					= (float)(levelFactor * xRange);
			float yScale 					= (float)(levelFactor * yRange);
				
			float ellipseRadiusX			= xScale;
			float ellipseRadiusY			= yScale;

			double minX						= anchorStartPoint.X - ellipseRadiusX;
			double maxX						= anchorStartPoint.X + ellipseRadiusX;
			DateTime minTime				= chartControl.GetTimeByX((int) minX);
			DateTime maxTime				= chartControl.GetTimeByX((int) maxX);

			// if its completely scrolled out of time range its not visible
			if (maxTime < firstTimeOnChart || minTime > lastTimeOnChart)
				return false;

			float minY = (float)anchorStartPoint.Y - ellipseRadiusY;
			float maxY = (float)anchorStartPoint.Y + ellipseRadiusY;

			// dont forget: smaller y = greater price value
			double minVal = chartScale.GetValueByY(maxY);
			double maxVal = chartScale.GetValueByY(minY);

			// completely off scale
			if (maxVal < chartScale.MinValue || minVal > chartScale.MaxValue)
				return false;

			return true;
		}

		public override void OnRender(ChartControl chartControl, ChartScale chartScale)
		{
			// nothing is drawn yet
			if (Anchors.All(a => a.IsEditing))
				return;

			ChartPanel chartPanel = chartControl.ChartPanels[PanelIndex];

			RenderTarget.AntialiasMode = SharpDX.Direct2D1.AntialiasMode.PerPrimitive;
			// get x distance of the line, this will be basis for our levels
			// unless extend left/right is also on
			Point anchorStartPoint 	= StartAnchor.GetPoint(chartControl, chartPanel, chartScale);
			Point anchorEndPoint 	= EndAnchor.GetPoint(chartControl, chartPanel, chartScale);
			AnchorLineStroke.RenderTarget = RenderTarget;
			
			SharpDX.Direct2D1.Brush tmpBrush = IsInHitTest ? chartControl.SelectionBrush : AnchorLineStroke.BrushDX;
			RenderTarget.DrawLine(anchorStartPoint.ToVector2(), anchorEndPoint.ToVector2(), tmpBrush, AnchorLineStroke.Width, AnchorLineStroke.StrokeStyle);
			
			// if we're doing a hit test pass, dont draw price levels at all, we dont want those to count for 
			// hit testing (match NT7)
			if (IsInHitTest || PriceLevels == null || !PriceLevels.Any())
				return;
			
			SetAllPriceLevelsRenderTarget();

			double indexRange = chartControl.GetSlotIndexByX(Convert.ToInt32(anchorEndPoint.X)) - chartControl.GetSlotIndexByX(Convert.ToInt32(anchorStartPoint.X));
			double ticksRange = (EndAnchor.Price - StartAnchor.Price)/chartControl.Instrument.MasterInstrument.TickSize;
			double radius = Math.Sqrt(Math.Pow(indexRange, 2) + Math.Pow(ticksRange, 2));
			double xRange = radius * chartControl.Properties.BarDistance ;
			double yRange = chartScale.GetPixelsForDistance(radius*chartControl.Instrument.MasterInstrument.TickSize);
			bool dirUp = anchorStartPoint.Y>anchorEndPoint.Y;
			
			if (IsDrawEllipse)
			{
		
				SharpDX.Direct2D1.EllipseGeometry lastEllipse = null;
				
				foreach (PT_PriceLevel priceLevel in PriceLevels.Where(pl => pl.IsVisible && pl.Stroke != null).OrderBy(pl => pl.Value))
				{
					float levelFactor 	= (float)priceLevel.Value/100f;
					float xScale 		= (float)(levelFactor * xRange);
					float yScale 		= (float)(levelFactor * yRange);
				
					// draw ellipse takes center point
					SharpDX.Vector2						startVec		= new SharpDX.Vector2((float)anchorStartPoint.X, (float)anchorStartPoint.Y);
					SharpDX.Direct2D1.Ellipse			ellipse			= new SharpDX.Direct2D1.Ellipse(startVec, xScale, yScale);
					SharpDX.Direct2D1.EllipseGeometry	ellipseGeometry	= new SharpDX.Direct2D1.EllipseGeometry(Core.Globals.D2DFactory, ellipse);

					RenderTarget.DrawEllipse(ellipse, priceLevel.Stroke.BrushDX, priceLevel.Stroke.Width, priceLevel.Stroke.StrokeStyle);

					Stroke backgroundStroke = new Stroke();
					priceLevel.Stroke.CopyTo(backgroundStroke);
					backgroundStroke.Opacity = PriceLevelOpacity;

					if (lastEllipse == null)
						RenderTarget.FillEllipse(ellipse, backgroundStroke.BrushDX);
					else
					{
						SharpDX.Direct2D1.GeometryGroup concentricCircles = new SharpDX.Direct2D1.GeometryGroup(Core.Globals.D2DFactory, SharpDX.Direct2D1.FillMode.Alternate, new SharpDX.Direct2D1.Geometry[] {lastEllipse, ellipseGeometry });
						RenderTarget.FillGeometry(concentricCircles, backgroundStroke.BrushDX);
						concentricCircles.Dispose();
					}
					
					lastEllipse = ellipseGeometry;
				}
				if (lastEllipse != null && !lastEllipse.IsDisposed)
					lastEllipse.Dispose();

			}
			else
			{
				SharpDX.Direct2D1.PathGeometry lastArc = null;
				float lastXScale = 0;
				float lastYScale = 0;
				
				foreach (PT_PriceLevel priceLevel in PriceLevels.Where(pl => pl.IsVisible && pl.Stroke != null).OrderBy(pl => pl.Value))
				{
					float levelFactor 	= (float)priceLevel.Value/100f;
					float xScale 		= (float)(levelFactor * xRange);
					float yScale 		= (float)(levelFactor * yRange);
					
						
					SharpDX.Direct2D1.PathGeometry arcGeometry = new SharpDX.Direct2D1.PathGeometry(Core.Globals.D2DFactory);
					SharpDX.Direct2D1.GeometrySink sink = arcGeometry.Open();
					Point pointStart = new Point (anchorStartPoint.X-xScale, anchorStartPoint.Y);
					Point pointEnd = new Point (anchorStartPoint.X+xScale,anchorStartPoint.Y);
					sink.BeginFigure(pointStart.ToVector2(), SharpDX.Direct2D1.FigureBegin.Filled);
					
					var arc_1 = new SharpDX.Direct2D1.ArcSegment()
					{
						ArcSize = SharpDX.Direct2D1.ArcSize.Small ,
				        RotationAngle = 180,
				        Point = pointEnd.ToVector2(),
				        Size = new SharpDX.Size2F(xScale, yScale),
				        SweepDirection = dirUp? SharpDX.Direct2D1.SweepDirection.Clockwise :
							SharpDX.Direct2D1.SweepDirection.CounterClockwise
					};
					sink.AddArc(arc_1);
					sink.EndFigure(SharpDX.Direct2D1.FigureEnd.Open);
					sink.Close();
					RenderTarget.DrawGeometry(arcGeometry, priceLevel.Stroke.BrushDX, priceLevel.Stroke.Width, priceLevel.Stroke.StrokeStyle);
					
					//Fillig
					Stroke backgroundStroke = new Stroke();
					priceLevel.Stroke.CopyTo(backgroundStroke);
					backgroundStroke.Opacity = PriceLevelOpacity;
					if (lastArc == null)
						RenderTarget.FillGeometry(arcGeometry, backgroundStroke.BrushDX);
					else
					{
						Point lastPointStart = new Point (anchorStartPoint.X-lastXScale, anchorStartPoint.Y);
						Point lastPointEnd = new Point (anchorStartPoint.X+lastXScale,anchorStartPoint.Y);
						// create geometry
						SharpDX.Direct2D1.PathGeometry fillingGeometry = new SharpDX.Direct2D1.PathGeometry(Core.Globals.D2DFactory);
						SharpDX.Direct2D1.GeometrySink sink2 = fillingGeometry.Open();
						
						
						sink2.BeginFigure(pointStart.ToVector2(), SharpDX.Direct2D1.FigureBegin.Filled);
						sink2.AddArc(arc_1);
						sink2.AddLine(lastPointEnd.ToVector2());
						var arc_2 = new SharpDX.Direct2D1.ArcSegment()
						{
							ArcSize = SharpDX.Direct2D1.ArcSize.Small ,
					        RotationAngle = 180,
					        Point = lastPointStart.ToVector2(),
					        Size = new SharpDX.Size2F(lastXScale, lastYScale),
					        SweepDirection = dirUp? SharpDX.Direct2D1.SweepDirection.CounterClockwise :
								SharpDX.Direct2D1.SweepDirection.Clockwise
						};
						sink2.AddArc(arc_2);
						sink2.AddLine(pointStart.ToVector2());
						sink2.EndFigure(SharpDX.Direct2D1.FigureEnd.Closed);
						sink2.Close();
						RenderTarget.FillGeometry(fillingGeometry, backgroundStroke.BrushDX);
						fillingGeometry.Dispose();	
					}				
					
					lastArc = arcGeometry;
					lastXScale = xScale;
					lastYScale = yScale;
				}
				if (lastArc != null && !lastArc.IsDisposed)
					lastArc.Dispose();
				
			}
		
			foreach (PT_PriceLevel priceLevel in PriceLevels.Where(pl => pl.IsVisible && pl.Stroke != null).OrderBy(pl => pl.Value))
			{
				SharpDX.Vector2 startVec = new SharpDX.Vector2((float)anchorStartPoint.X, (float)anchorStartPoint.Y);
				float levelFactor 	= (float)priceLevel.Value/100f;
				float yScale 		= (float)(levelFactor * yRange);
				float textX				= startVec.X;
				float textY;
				double yVal = StartAnchor.Price + (EndAnchor.Price - StartAnchor.Price) * levelFactor;
				if (dirUp)
					textY = startVec.Y - yScale;
				else
					textY = startVec.Y + yScale;
				DrawPriceLevelText(textX, textY, priceLevel, yVal, chartControl);
			}
		}

		protected override void OnStateChange()
		{
			if (State == State.SetDefaults)
			{
				AnchorLineStroke 				= new Stroke(Brushes.DarkGray, DashStyleHelper.Solid, 1f, 50);
				Name 							= "Fibonacci Arc (PT)";
				PriceLevelOpacity				= 5;
				StartAnchor						= new ChartAnchor { IsEditing = true, DrawingTool = this };
				EndAnchor						= new ChartAnchor { IsEditing = true, DrawingTool = this };
				StartAnchor.DisplayName			= Custom.Resource.NinjaScriptDrawingToolAnchorStart;
				EndAnchor.DisplayName			= Custom.Resource.NinjaScriptDrawingToolAnchorEnd;
				IsTextDisplayed					= true;
				IsDrawEllipse 					= false;
			}
			else if (State == State.Configure)
			{
				if (PriceLevels.Count == 0)
				{
					PriceLevels.Add(new PT_PriceLevel(38.2, 	Brushes.DodgerBlue));
					PriceLevels.Add(new PT_PriceLevel(61.8, 	Brushes.CornflowerBlue));
					PriceLevels.Add(new PT_PriceLevel(100, 	Brushes.SteelBlue));
				}
			}
			else if (State == State.Terminated)
				Dispose();
		}
	}

	/// <summary>
	/// Represents an interface that exposes information regarding a Fibonacci Fan IDrawingTool.
	/// </summary>
	[EditorBrowsable(EditorBrowsableState.Always)]
	public class PT_FibonacciFan : PT_FibonacciLevels
	{
		internal struct LevelAlertData
		{
			public PT_AlertCondition condition;
			public PT_AlertRearmOn rearm;
			public bool	enabled;
			public bool playSound;
			public bool showPopup;
			public string sound;
			public int rearmTime;
			public DateTime lastBarTime;
			public bool armed;
			public DateTime armedTime;
		}

		[Display(ResourceType = typeof(Custom.Resource), Name = "ExtendLines", GroupName = "NinjaScriptLines")]
		public bool 					IsExtendedLines 	{ get; set; }

		[Display(ResourceType = typeof(Custom.Resource), Name = "TextLocation", GroupName = "NinjaScriptGeneral")]
		public PT_TextLocation				TextLocation { get; set; }
		
		
		protected bool CheckAlertFanLine(Condition condition, Point lineStartPoint, Point lineEndPoint,
													ChartControl chartControl, ChartScale chartScale, ChartAlertValue[] values)
		{
			// not completely drawn yet?
			if (Anchors.Count(a => a.IsEditing) > 1)
				return false;

			if (values[0].ValueType == ChartAlertValueType.StaticTime)
			{
				int checkX = chartControl.GetXByTime(values[0].Time);
				return lineStartPoint.X >= checkX || lineEndPoint.X >= checkX;
			}

			double firstBarX	= chartControl.GetXByTime(values[0].Time);
			double firstBarY	= chartScale.GetYByValue(values[0].Value);
			Point barPoint		= new Point(firstBarX, firstBarY);

			 // bars passed our drawing tool line
			if (lineEndPoint.X < firstBarX)
				return false;	

			// bars not yet to our drawing tool line
			if (lineStartPoint.X > firstBarX)
				return false;

			// NOTE: 'left / right' is relative to if line was vertical. it can end up backwards too
			MathHelper.PointLineLocation pointLocation = MathHelper.GetPointLineLocation(lineStartPoint, lineEndPoint, barPoint);
			// for vertical things, think of a vertical line rotated 90 degrees to lay flat, where it's normal vector is 'up'
			switch (condition)
			{
				case Condition.Greater:			return pointLocation == MathHelper.PointLineLocation.LeftOrAbove;
				case Condition.GreaterEqual:	return pointLocation == MathHelper.PointLineLocation.LeftOrAbove || pointLocation == MathHelper.PointLineLocation.DirectlyOnLine;
				case Condition.Less:			return pointLocation == MathHelper.PointLineLocation.RightOrBelow;
				case Condition.LessEqual:		return pointLocation == MathHelper.PointLineLocation.RightOrBelow || pointLocation == MathHelper.PointLineLocation.DirectlyOnLine;
				case Condition.Equals:			return pointLocation == MathHelper.PointLineLocation.DirectlyOnLine;
				case Condition.NotEqual:		return pointLocation != MathHelper.PointLineLocation.DirectlyOnLine;
				case Condition.CrossAbove:
				case Condition.CrossBelow:
					Predicate<ChartAlertValue> predicate = v =>
					{
						if (v.Time == Core.Globals.MinDate)
							return false;
						double barX = chartControl.GetXByTime(v.Time);
						double barY = chartScale.GetYByValue(v.Value);
						Point stepBarPoint = new Point(barX, barY);
						// NOTE: 'left / right' is relative to if line was vertical. it can end up backwards too
						MathHelper.PointLineLocation ptLocation = MathHelper.GetPointLineLocation(lineStartPoint, lineEndPoint, stepBarPoint);
						if (condition == Condition.CrossAbove)
							return ptLocation == MathHelper.PointLineLocation.LeftOrAbove;
						return ptLocation == MathHelper.PointLineLocation.RightOrBelow;
					};
					return MathHelper.DidPredicateCross(values, predicate);
			}

			return false;
		}

		protected bool CheckInternalAlertFanLine(PT_AlertCondition condition, Point lineStartPoint, Point lineEndPoint,
													ChartControl chartControl, ChartScale chartScale, double price, DateTime time, double prevPrice, DateTime prevTime)
		{
			// not completely drawn yet?
			if (Anchors.Count(a => a.IsEditing) > 1)
				return false;

			double firstBarX	= chartControl.GetXByTime(time);
			double firstBarY	= chartScale.GetYByValue(price);
			Point barPoint		= new Point(firstBarX, firstBarY);

			 // bars passed our drawing tool line
			if (lineEndPoint.X < firstBarX)
				return false;	

			// bars not yet to our drawing tool line
			if (lineStartPoint.X > firstBarX)
				return false;

			// NOTE: 'left / right' is relative to if line was vertical. it can end up backwards too
			MathHelper.PointLineLocation pointLocation = MathHelper.GetPointLineLocation(lineStartPoint, lineEndPoint, barPoint);
			// for vertical things, think of a vertical line rotated 90 degrees to lay flat, where it's normal vector is 'up'
			double barX = chartControl.GetXByTime(time);
			double barY = chartScale.GetYByValue(price);
			double barXprev = chartControl.GetXByTime(prevTime);
			double barYprev = chartScale.GetYByValue(prevPrice);
			Point stepBarPoint = new Point(barX, barY);
			Point prevBarPoint = new Point(barXprev, barYprev);
			MathHelper.PointLineLocation ptLocation = MathHelper.GetPointLineLocation(lineStartPoint, lineEndPoint, stepBarPoint);
			MathHelper.PointLineLocation prevPtLocation = MathHelper.GetPointLineLocation(lineStartPoint, lineEndPoint, prevBarPoint);
			switch (condition)
			{
				case PT_AlertCondition.Equal:		return pointLocation == MathHelper.PointLineLocation.DirectlyOnLine;
				case PT_AlertCondition.CrossAbove:
					return (ptLocation== MathHelper.PointLineLocation.LeftOrAbove && prevPtLocation == MathHelper.PointLineLocation.RightOrBelow);
				case PT_AlertCondition.CrossBelow:
					return (prevPtLocation== MathHelper.PointLineLocation.LeftOrAbove && ptLocation == MathHelper.PointLineLocation.RightOrBelow);
				case PT_AlertCondition.Any:
					return ptLocation==MathHelper.PointLineLocation.DirectlyOnLine ||
						(ptLocation== MathHelper.PointLineLocation.LeftOrAbove && prevPtLocation == MathHelper.PointLineLocation.RightOrBelow) ||
						(prevPtLocation== MathHelper.PointLineLocation.LeftOrAbove && ptLocation == MathHelper.PointLineLocation.RightOrBelow);
			}
			return false;
		}

		
		protected void DrawPriceLevelText(ChartPanel chartPanel, ChartScale chartScale, double startX, double endX, double y, double price, PT_PriceLevel priceLevel)
		{
			if (TextLocation == PT_TextLocation.Off || priceLevel == null || priceLevel.Stroke == null || priceLevel.Stroke.BrushDX == null)
				return;
			
			// make a rectangle that sits right at our line, depending on text alignment settings
			SimpleFont						wpfFont		= chartPanel.ChartControl.Properties.LabelFont ?? new SimpleFont();
			SharpDX.DirectWrite.TextFormat	textFormat	= wpfFont.ToDirectWriteTextFormat();
			textFormat.TextAlignment					= SharpDX.DirectWrite.TextAlignment.Leading;
			textFormat.WordWrapping						= SharpDX.DirectWrite.WordWrapping.NoWrap;
			
			string str		= GetPriceString(price, priceLevel);

			// when using extreme alignments, give a few pixels of padding on the text so we dont end up right on the edge
			const double	edgePadding	= 2f;
			float			layoutWidth	= (float)Math.Abs(startX - endX); // always give entire available width for layout
			// dont use max x for max text width here, that can break inside left/right when extended lines are on
			SharpDX.DirectWrite.TextLayout textLayout = new SharpDX.DirectWrite.TextLayout(Core.Globals.DirectWriteFactory, str, textFormat, layoutWidth, textFormat.FontSize);

			double drawAtX;
				
			if (TextLocation == PT_TextLocation.Extreme)
			{
				if (startX < endX)
					drawAtX = endX - 1;
				else
					drawAtX = endX - 1 - textLayout.Metrics.Width;
			}
			else
			{
				if (startX > endX)
					drawAtX = endX - 1;
				else
					drawAtX = endX - 1 - textLayout.Metrics.Width;
			}


			// we also move our y value up by text height so we draw label above line like NT7.
			RenderTarget.DrawTextLayout(new SharpDX.Vector2((float)drawAtX, (float)(y - textFormat.FontSize - edgePadding)),  textLayout, priceLevel.Stroke.BrushDX, SharpDX.Direct2D1.DrawTextOptions.NoSnap);

			textFormat.Dispose();
			textLayout.Dispose();
		}
		
		public override Cursor GetCursor(ChartControl chartControl, ChartPanel chartPanel, ChartScale chartScale, Point point)
		{
			switch (DrawingState)
			{
				case DrawingState.Building:	return Cursors.Pen;
				case DrawingState.Moving:	return IsLocked ? Cursors.No : Cursors.SizeAll;
				case DrawingState.Editing:
					if (IsLocked)
						return Cursors.No;
					return editingAnchor == StartAnchor ? Cursors.SizeNESW : Cursors.SizeNWSE;
				default:
					// draw move cursor if cursor is near line path anywhere
					Point startAnchorPixelPoint	= StartAnchor.GetPoint(chartControl, chartPanel, chartScale);

					ChartAnchor closest = GetClosestAnchor(chartControl, chartPanel, chartScale, CursorSensitivity, point);
					if (closest != null)
					{
						if (IsLocked)
							return null;
						return closest == StartAnchor ? Cursors.SizeNESW : Cursors.SizeNWSE;
					}

					Vector	totalVector	= EndAnchor.GetPoint(chartControl, chartPanel, chartScale) - startAnchorPixelPoint;
					return MathHelper.IsPointAlongVector(point, startAnchorPixelPoint, totalVector, CursorSensitivity) ? 
						IsLocked ? Cursors.Arrow : Cursors.SizeAll :
						null;
			}
		}
		
		public Point CalculateExtendedDataPoint(ChartPanel panel, ChartScale scale, int startX, double startY, Vector slope)
		{
			// find direction of slope
			bool	right	= slope.X > 0;
			bool	up		= slope.Y < 0;
			int		xLength = right ? panel.W - startX : panel.X + startX;
			double	priceSize = Math.Abs(xLength / slope.X) * slope.Y;
			double	endY = startY + priceSize;
			double	maxY = up ? scale.GetYByValue(panel.MaxValue) : scale.GetYByValue(panel.MinValue);
			//return new Point(right ? panel.W : 0, maxY);
			// check if Y endpoint is outside top or bottom of panel
			if (up ? endY < maxY : maxY < endY)
			{
				double yLength = Math.Abs(maxY - startY);
				double xSize = Math.Abs(yLength / slope.Y) * slope.X;
				double endX = startX + xSize;
				return new Point(endX, maxY);
			}
			else
			{
				return new Point(right ? panel.W : 0, endY);
			}
		}

		// Item1 = leftmost point, Item2 rightmost point of line
		protected Tuple<Point, Point> GetPriceLevelLinePoints(PT_PriceLevel priceLevel, ChartControl chartControl, ChartScale chartScale, bool isInverted)
		{
			ChartPanel chartPanel	= chartControl.ChartPanels[PanelIndex];
			Point anchorStartPoint 	= StartAnchor.GetPoint(chartControl, chartPanel, chartScale);
			Point anchorEndPoint 	= EndAnchor.GetPoint(chartControl, chartPanel, chartScale);
			double totalPriceRange 	= EndAnchor.Price - StartAnchor.Price;
			double lineStartX		= anchorStartPoint.X;
			double lineEndX 		= anchorEndPoint.X;
			double lineStartY		= anchorStartPoint.Y;
			double lineEndY			= priceLevel.GetY(chartScale, StartAnchor.Price, totalPriceRange, isInverted);
			if (IsExtendedLines)	
			{
				Vector slope = new Vector();
				slope.X = anchorEndPoint.X - anchorStartPoint.X;
				slope.Y =  lineEndY - anchorStartPoint.Y;
				Point lineEndPoint = CalculateExtendedDataPoint(chartPanel, chartScale, Convert.ToInt32(anchorStartPoint.X), anchorStartPoint.Y, slope);
				lineEndX = lineEndPoint.X;
				lineEndY = lineEndPoint.Y;
			}
			return new Tuple<Point, Point>(new Point(lineStartX, lineStartY), new Point(lineEndX, lineEndY));
		}

		private string GetPriceString(double price, PT_PriceLevel priceLevel)
		{
			// note, dont use MasterInstrument.FormatPrice() as it will round value to ticksize which we do not want
			string priceStr	= price.ToString(Core.Globals.GetTickFormatString(AttachedTo.Instrument.MasterInstrument.TickSize));
			double pct		= priceLevel.Value / 100d;
			string str		= string.Format("{0} ({1})", pct.ToString("P", Core.Globals.GeneralOptions.CurrentCulture), priceStr);
			return str;
		}

		private string GetPercentString(PT_PriceLevel priceLevel)
		{
			double pct		= priceLevel.Value / 100d;
			string str		= string.Format("{0}", pct.ToString("P", Core.Globals.GeneralOptions.CurrentCulture));
			return str;
		}
		
		public override Point[] GetSelectionPoints(ChartControl chartControl, ChartScale chartScale)
		{
			ChartPanel chartPanel = chartControl.ChartPanels[PanelIndex];
			
			Point startPoint 	= StartAnchor.GetPoint(chartControl, chartPanel, chartScale);
			Point endPoint 		= EndAnchor.GetPoint(chartControl, chartPanel, chartScale);
			Point midPoint		= new Point((startPoint.X + endPoint.X) / 2, (startPoint.Y + endPoint.Y) / 2);
			
			return new[] { startPoint, midPoint, endPoint };
		}
		
		public override bool IsAlertConditionTrue(AlertConditionItem conditionItem, Condition condition, ChartAlertValue[] values, ChartControl chartControl, ChartScale chartScale)
		{
			
			PT_PriceLevel priceLevel = conditionItem.Tag as PT_PriceLevel;
			if (priceLevel == null)
				return false;
			Tuple<Point, Point>	plp = GetPriceLevelLinePoints(priceLevel, chartControl, chartScale, true);
			return CheckAlertFanLine(condition, plp.Item1, plp.Item2, chartControl, chartScale, values);
		
		}

		public override bool IsVisibleOnChart(ChartControl chartControl, ChartScale chartScale, DateTime firstTimeOnChart, DateTime lastTimeOnChart)
		{
			return true;
		}

		public bool IsReallyVisibleOnChart(ChartControl chartControl, ChartScale chartScale, DateTime firstTimeOnChart, DateTime lastTimeOnChart)
		{
			if (DrawingState == DrawingState.Building)
				return true;

			DateTime minTime = Core.Globals.MaxDate;
			DateTime maxTime = Core.Globals.MinDate;
			foreach (ChartAnchor anchor in Anchors)
			{
				if (anchor.Time < minTime)
					minTime = anchor.Time;
				if (anchor.Time > maxTime)
					maxTime = anchor.Time;
			}

			if (!IsExtendedLines)
				return new[]{minTime,maxTime}.Any(t => t >= firstTimeOnChart && t <= lastTimeOnChart) || (minTime < firstTimeOnChart && maxTime > lastTimeOnChart);

			return true;
		}

		public override void OnCalculateMinMax()
		{
			MinValue = double.MaxValue;
			MaxValue = double.MinValue;

			if (!IsVisible)
				return;

			// make sure *something* is drawn yet, but dont blow up if editing just a single anchor
			if (Anchors.All(a => a.IsEditing))
				return;

			double totalPriceRange 	= EndAnchor.Price - StartAnchor.Price;
			double startPrice = StartAnchor.Price;// + yPriceOffset;
			foreach (PT_PriceLevel priceLevel in PriceLevels.Where(pl => pl.IsVisible && pl.Stroke != null))
			{
				double levelPrice	= startPrice + (1 - priceLevel.Value/100) * totalPriceRange;
				MinValue = Math.Min(MinValue, levelPrice);
				MaxValue = Math.Max(MaxValue, levelPrice);
			}
		}

		public override void OnMouseDown(ChartControl chartControl, ChartPanel chartPanel, ChartScale chartScale, ChartAnchor dataPoint)
		{
			switch (DrawingState)
			{
				case DrawingState.Building:
					if (StartAnchor.IsEditing)
					{
						dataPoint.CopyDataValues(StartAnchor);
						// give end anchor something to start with so we dont try to render it with bad values right away
						dataPoint.CopyDataValues(EndAnchor);
						StartAnchor.IsEditing = false;
					}
					else if (EndAnchor.IsEditing)
					{
						dataPoint.CopyDataValues(EndAnchor);
						EndAnchor.IsEditing = false;
					}
					
					// is initial building done (both anchors set)
					if (!StartAnchor.IsEditing && !EndAnchor.IsEditing)
					{
						DrawingState = DrawingState.Normal;
						IsSelected = false; 
					}
					break;
				case DrawingState.Normal:
					Point point = dataPoint.GetPoint(chartControl, chartPanel, chartScale);
					editingAnchor = GetClosestAnchor(chartControl, chartPanel, chartScale, CursorSensitivity, point);
					if (editingAnchor != null)
					{
						editingAnchor.IsEditing = true;
						DrawingState = DrawingState.Editing;
					}
					else if (editingAnchor == null || IsLocked)
					{
						// or if they didnt click particulary close to either, move (they still clicked close to our line)
						// set it to moving even if locked so we know to change cursor
						if (GetCursor(chartControl, chartPanel, chartScale, point) != null)
							DrawingState = DrawingState.Moving;
						else
							IsSelected = false;
					}
					break;
			}
		}
		
		public override void OnMouseMove(ChartControl chartControl, ChartPanel chartPanel, ChartScale chartScale, ChartAnchor dataPoint)
		{
			if (IsLocked && DrawingState != DrawingState.Building)
				return;

			if (DrawingState == DrawingState.Building)
			{
				// start anchor will not be editing here because we start building as soon as user clicks, which
				// plops down a start anchor right away
				if (EndAnchor.IsEditing)
					dataPoint.CopyDataValues(EndAnchor);
			}
			else if (DrawingState == DrawingState.Editing && editingAnchor != null)
				dataPoint.CopyDataValues(editingAnchor);
			else if (DrawingState == DrawingState.Moving)
				foreach (ChartAnchor anchor in Anchors)
					anchor.MoveAnchor(InitialMouseDownAnchor, dataPoint, chartControl, chartPanel, chartScale, this);
		}
		
		public override void OnMouseUp(ChartControl chartControl, ChartPanel chartPanel, ChartScale chartScale, ChartAnchor dataPoint)
		{
			// simply end whatever moving
			if (DrawingState == DrawingState.Editing || DrawingState == DrawingState.Moving)
				DrawingState = DrawingState.Normal;
			if (editingAnchor != null)
				editingAnchor.IsEditing = false;
			editingAnchor = null;
		}
		
		void ShowAlertMessage(ChartControl chartControl, string msg)
		{
			chartControl.Dispatcher.InvokeAsync(new Action(() => {      
				NinjaTrader.Gui.Tools.NTMessageBoxSimple.Show(Window.GetWindow(chartControl.OwnerChart as DependencyObject), msg, "Alert", MessageBoxButton.OK, MessageBoxImage.Exclamation);
			}));
		}
		
		public string GetTFName (ChartControl chartControl)
		{
			ChartBars chartBars= GetAttachedToChartBars();
			if (chartBars.Bars.BarsPeriod.BarsPeriodType == BarsPeriodType.Minute)
			{
				switch (chartBars.Bars.BarsPeriod.Value)
				{
					case 1: return ("M1");
					case 5: return ("M5");      
					case 15: return ("M15");
					case 30: return ("M30");
					case 60: return ("H1");
					case 240: return ("H4");
				}
				return("M" + chartBars.Bars.BarsPeriod.Value.ToString());
			}
			
			if (chartBars.Bars.BarsPeriod.BarsPeriodType == BarsPeriodType.Day)
			{
				return("D" + chartBars.Bars.BarsPeriod.Value.ToString());
			}
			
			if (chartBars.Bars.BarsPeriod.BarsPeriodType == BarsPeriodType.Week)
			{
				return("W" + chartBars.Bars.BarsPeriod.Value.ToString());
			}
			
			if (chartBars.Bars.BarsPeriod.BarsPeriodType == BarsPeriodType.Month)
			{
				return("MN" + chartBars.Bars.BarsPeriod.Value.ToString());
			}
			return(chartBars.Bars.BarsPeriod.BarsPeriodType.ToString() + " " + chartBars.Bars.BarsPeriod.Value.ToString());
		}

		void ProcessAlerts(ChartControl chartControl, ChartScale chartScale)
		{	
			ChartBars chartBars= GetAttachedToChartBars();
			if (chartBars.Bars.Count<2)
				return;
			double price = chartBars.Bars.GetClose(chartBars.Bars.Count-1);
			DateTime time = chartBars.Bars.GetTime(chartBars.Bars.Count-1);
			double prevPrice = chartBars.Bars.GetClose(chartBars.Bars.Count-2);
			DateTime prevTime = chartBars.Bars.GetTime(chartBars.Bars.Count-2);

			int i=0;
			foreach (PT_PriceLevel priceLevel in PriceLevels)
			{
				if (priceLevel.AlertEnabled)
				{
					Tuple<Point, Point>	plp = GetPriceLevelLinePoints(priceLevel, chartControl, chartScale, true);
					if (CheckInternalAlertFanLine(priceLevel.AlertCondition, plp.Item1, plp.Item2, chartControl, chartScale, price, time, prevPrice, prevTime))
					{
						String msg = chartControl.Instrument + ", "+ GetTFName(chartControl)+", "+this.Tag+", Level "+priceLevel.Value+"%) alert by "+ " "+ priceLevel.AlertCondition + " condition";
						int rearmSec = (priceLevel.AlertRearm == PT_AlertRearmOn.OnTimer)? priceLevel.AlertRearmTime:int.MaxValue;
						string sndFile = (priceLevel.AlertPlaySound)?NinjaTrader.Core.Globals.InstallDir+@"\sounds\"+ priceLevel.AlertSound:"";
						if (!priceLevel.AlertArmed || 
							priceLevel.AlertRearm==PT_AlertRearmOn.OnTimer && (DateTime.Now-priceLevel.AlertArmedTime)>TimeSpan.FromSeconds(priceLevel.AlertRearmTime))
						{
							DateTime now = (Cbi.Connection.PlaybackConnection != null ? Cbi.Connection.PlaybackConnection.Now : Core.Globals.Now);
							NinjaTrader.NinjaScript.Alert.AlertCallback(chartControl.Instrument, this, this.Tag+", Level "+priceLevel.Value, now, Priority.High, msg, sndFile, null, null, rearmSec);
							priceLevel.AlertArmed = true;
							priceLevel.AlertArmedTime = DateTime.Now;
							if (priceLevel.AlertShowPopup)
								ShowAlertMessage(chartControl, msg);
						}							
					}
					else
					{
						if (priceLevel.AlertArmed)
						{
							if ((priceLevel.AlertRearm==PT_AlertRearmOn.BarClose && priceLevel.AlertLastBarTime != time) ||
								(priceLevel.AlertRearm == PT_AlertRearmOn.ConditionReversed) ||
								priceLevel.AlertRearm==PT_AlertRearmOn.OnTimer && (DateTime.Now-priceLevel.AlertArmedTime)>TimeSpan.FromSeconds(priceLevel.AlertRearmTime))
							{
								NinjaTrader.NinjaScript.Alert.RearmAlert(this.Tag+", Level "+(i+1));
								priceLevel.AlertArmed = false;
							}
						}
					}
					
					priceLevel.AlertLastBarTime = time;
				}
				i++;
			}
		}
		
		public override void OnRender(ChartControl chartControl, ChartScale chartScale)
		{
			// nothing is drawn yet
			if (Anchors.All(a => a.IsEditing))
				return;
		
			ProcessAlerts(chartControl, chartScale);
			
			if (!IsReallyVisibleOnChart(chartControl, chartScale, chartControl.FirstTimePainted, chartControl.LastTimePainted))
				return;
				
			
			RenderTarget.AntialiasMode			= SharpDX.Direct2D1.AntialiasMode.PerPrimitive;
			ChartPanel chartPanel				= chartControl.ChartPanels[PanelIndex];
			// get x distance of the line, this will be basis for our levels
			// unless extend left/right is also on
			Point anchorStartPoint 				= StartAnchor.GetPoint(chartControl, chartPanel, chartScale);
			Point anchorEndPoint 				= EndAnchor.GetPoint(chartControl, chartPanel, chartScale);
			
			AnchorLineStroke.RenderTarget		= RenderTarget;
			
			SharpDX.Direct2D1.Brush tmpBrush	= IsInHitTest ? chartControl.SelectionBrush : AnchorLineStroke.BrushDX;
			RenderTarget.DrawLine(anchorStartPoint.ToVector2(), anchorEndPoint.ToVector2(), tmpBrush, AnchorLineStroke.Width, AnchorLineStroke.StrokeStyle);
			
			// if we're doing a hit test pass, dont draw price levels at all, we dont want those to count for 
			// hit testing (match NT7)
			if (IsInHitTest || PriceLevels == null || !PriceLevels.Any())
				return;
			
			SetAllPriceLevelsRenderTarget();

			Point	lastStartPoint	= new Point(0, 0);
			Point	lastEndPoint	= new Point(0, 0);
			Stroke	lastStroke		= null;

			foreach (PT_PriceLevel priceLevel in PriceLevels.Where(pl => pl.IsVisible && pl.Stroke != null).OrderBy(p=>p.Value))
			{
				Tuple<Point, Point>	plp = GetPriceLevelLinePoints(priceLevel, chartControl, chartScale, true);

				// align to full pixel to avoid unneeded aliasing
				double strokePixAdj =	(priceLevel.Stroke.Width % 2.0).ApproxCompare(0) == 0 ? 0.5d : 0d;
				Vector pixelAdjustVec = new Vector(strokePixAdj, strokePixAdj);
			
				RenderTarget.DrawLine(anchorStartPoint.ToVector2(), (plp.Item2 + pixelAdjustVec).ToVector2(),
										priceLevel.Stroke.BrushDX, priceLevel.Stroke.Width, priceLevel.Stroke.StrokeStyle);

				if (lastStroke == null)
					lastStroke = new Stroke();
				else
				{
					// create geometry
					SharpDX.Direct2D1.PathGeometry lineGeometry = new SharpDX.Direct2D1.PathGeometry(Core.Globals.D2DFactory);
					SharpDX.Direct2D1.GeometrySink sink = lineGeometry.Open();
					sink.BeginFigure(anchorStartPoint.ToVector2(), SharpDX.Direct2D1.FigureBegin.Filled);
					Point  extendedEndPoint = (plp.Item2 + pixelAdjustVec);
					sink.AddLine(lastEndPoint.ToVector2());
					// Does the fill color need to fill a corner?  Check and add a point
					
					if (Math.Abs(lastEndPoint.Y - extendedEndPoint.Y) > 0.1 && Math.Abs(lastEndPoint.X - extendedEndPoint.X) > 0.1)
					{
						double boundaryX;
						double boundaryY;
						boundaryY = lastEndPoint.Y;
						boundaryX = extendedEndPoint.X;
						sink.AddLine(new SharpDX.Vector2((float)boundaryX, (float)boundaryY));
					}					
					sink.AddLine((plp.Item2 + pixelAdjustVec).ToVector2());
					sink.AddLine(anchorStartPoint.ToVector2());
					
					sink.EndFigure(SharpDX.Direct2D1.FigureEnd.Closed);
					sink.Close();
					RenderTarget.FillGeometry(lineGeometry, lastStroke.BrushDX);
					lineGeometry.Dispose();	
				}
				priceLevel.Stroke.CopyTo(lastStroke);
				lastStroke.Opacity	= PriceLevelOpacity;
				lastStartPoint		= plp.Item1 + pixelAdjustVec;
				lastEndPoint		= plp.Item2 + pixelAdjustVec;
			}

			// Render price text after background colors have rendered so the price text is on top
			foreach (PT_PriceLevel priceLevel in PriceLevels.Where(pl => pl.IsVisible && pl.Stroke != null))
			{
				if (!IsExtendedLines)
				{
					// dont always draw the text at min/max x the line renders at, pass anchor min max
					// in case text alignment is not extreme
					float	plPixAdjust		= (priceLevel.Stroke.Width % 2.0).ApproxCompare(0) == 0 ? 0.5f : 0f;
					double totalPriceRange 	= EndAnchor.Price - StartAnchor.Price;
					double price			= priceLevel.GetPrice(StartAnchor.Price, totalPriceRange, true);
					double textY = priceLevel.GetY(chartScale, StartAnchor.Price, totalPriceRange, true);
					DrawPriceLevelText(chartPanel, chartScale, anchorStartPoint.X, anchorEndPoint.X+plPixAdjust, textY, price, priceLevel);
				}
				else 
				{
					if (TextLocation == PT_TextLocation.Off)
						break;
					double totalPriceRange 	= EndAnchor.Price - StartAnchor.Price;
					Tuple<Point, Point>	plp2  = GetPriceLevelLinePoints(priceLevel, chartControl, chartScale, true);
					SimpleFont						wpfFont		= chartControl.Properties.LabelFont ?? new SimpleFont();
					SharpDX.DirectWrite.TextFormat	textFormat	= wpfFont.ToDirectWriteTextFormat();
					textFormat.TextAlignment					= SharpDX.DirectWrite.TextAlignment.Leading;
					textFormat.WordWrapping						= SharpDX.DirectWrite.WordWrapping.NoWrap;
					string							str			= GetPercentString(priceLevel);
					SharpDX.DirectWrite.TextLayout	textLayout	= new SharpDX.DirectWrite.TextLayout(Core.Globals.DirectWriteFactory, str, textFormat, 100, textFormat.FontSize);

					// once text is laid out, update used width to calcuated space required
					float fontHeight		= textLayout.Metrics.Height;

					Point textEndPoint		= new Point(plp2.Item2.X, plp2.Item2.Y);
					ChartPanel	panel			= chartControl.ChartPanels[PanelIndex];
					
					if (totalPriceRange > 0)
					{						
						textEndPoint.Y = textEndPoint.Y - fontHeight ;
					}
					
					if (textEndPoint.X > panel.X + panel.W - textLayout.Metrics.Width)
					{
						textEndPoint.X = panel.X + panel.W - textLayout.Metrics.Width;
					}

					if (totalPriceRange > 0)
					{
						if (textEndPoint.Y < panel.Y + (fontHeight * 0.5))
							textEndPoint.Y = panel.Y + (fontHeight * 0.5);
					}
					else
					{
						if (textEndPoint.Y > panel.Y + panel.H - (fontHeight * 1.5))
							textEndPoint.Y = panel.Y + panel.H - (fontHeight * 1.5);
					}

					float?	marginResource	= Application.Current.FindResource("FontModalTitleMargin") as float?;
					float	margin			= 2f + (marginResource.HasValue ? marginResource.Value : 3f);
					// Allow for changes in X position based on whether text is aligned to left or right edge of screen
					
					float	marginX			= (EndAnchor.Time < StartAnchor.Time) ? margin : -2 * margin;

					SharpDX.Vector2		endVec			= new SharpDX.Vector2((float) textEndPoint.X, (float) textEndPoint.Y);
					SharpDX.Matrix3x2	transformMatrix	= SharpDX.Matrix3x2.Translation(endVec);

					RenderTarget.Transform				= transformMatrix;

					RenderTarget.DrawTextLayout(new SharpDX.Vector2(marginX + margin, margin), textLayout, priceLevel.Stroke.BrushDX, SharpDX.Direct2D1.DrawTextOptions.NoSnap);

					RenderTarget.Transform				= SharpDX.Matrix3x2.Identity;
					textFormat.Dispose();
					textLayout.Dispose();
				}
			}
		}
	
		protected override void OnStateChange()
		{
			if (State == State.SetDefaults)
			{
				AnchorLineStroke 			= new Stroke(Brushes.DarkGray, DashStyleHelper.Solid, 1f, 50);
				Name 						= "Fibonacci Fan (PT)";
				PriceLevelOpacity			= 5;
				StartAnchor					= new ChartAnchor { IsEditing = true, DrawingTool = this };
				EndAnchor					= new ChartAnchor { IsEditing = true, DrawingTool = this };
				StartAnchor.DisplayName		= Custom.Resource.NinjaScriptDrawingToolAnchorStart;
				EndAnchor.DisplayName		= Custom.Resource.NinjaScriptDrawingToolAnchorEnd;
				TextLocation = PT_TextLocation.Extreme;
								
			}
			else if (State == State.Configure)
			{
				if (PriceLevels.Count == 0)
				{	
					PriceLevels.Add(new PT_PriceLevel(23.6,	Brushes.DarkGray));
					PriceLevels.Add(new PT_PriceLevel(27.2,	Brushes.DodgerBlue));
					PriceLevels.Add(new PT_PriceLevel(38.2,	Brushes.CornflowerBlue));
					PriceLevels.Add(new PT_PriceLevel(50.0,		Brushes.SteelBlue));
					PriceLevels.Add(new PT_PriceLevel(61.8,	Brushes.DarkCyan));
					PriceLevels.Add(new PT_PriceLevel(76.4,	Brushes.SeaGreen));
					PriceLevels.Add(new PT_PriceLevel(78.6,	Brushes.DarkGray));
				}
				
				foreach (PT_PriceLevel priceLevel in PriceLevels)
				{
					
					NinjaTrader.NinjaScript.Alert.RearmAlert(this.Tag+", Level "+ priceLevel.Value);
				}

			}
			else if (State == State.Terminated)
				Dispose();
		}
	}


	// when creating a custom type converter for drawing tools it must inherit from DrawingToolsPropertyConverter to work correctly with chart anchors
	public class PT_FibonacciCircleTimeTypeConverter : Gui.DrawingTools.DrawingToolPropertiesConverter
	{
		public override bool GetPropertiesSupported(ITypeDescriptorContext context)
		{
			return true;
		}

		public override PropertyDescriptorCollection GetProperties(ITypeDescriptorContext context, object value, Attribute[] attributes)
		{
			// override the GetProperties method to return a property collection that hides extended lines & text alignment properties
			// since they do not apply to fib circle / time
			PropertyDescriptorCollection propertyDescriptorCollection	= base.GetPropertiesSupported(context) ? base.GetProperties(context, value, attributes) : TypeDescriptor.GetProperties(value, attributes);
			PropertyDescriptorCollection adjusted						= new PropertyDescriptorCollection(null);
			if (propertyDescriptorCollection != null)
				foreach (PropertyDescriptor property in propertyDescriptorCollection)
					if (property.Name != "IsExtendedLinesRight" && property.Name != "IsExtendedLinesLeft" && property.Name != "TextLocation")
						adjusted.Add(property);
			return adjusted;
		}
	}
/*
	public static partial class Draw
	{
		private static T PT_FibonacciCore<T>(NinjaScriptBase owner, bool isAutoScale, string tag,
			int startBarsAgo, DateTime startTime, double startY, 
			int endBarsAgo, DateTime endTime, double endY, bool isGlobal, string templateName) where T : FibonacciLevels
		{
			if (owner == null)
				throw new ArgumentException("owner");
			if (startTime == Core.Globals.MinDate && endTime == Core.Globals.MinDate && startBarsAgo == int.MinValue && endBarsAgo == int.MinValue)
				throw new ArgumentException("bad start/end date/time");

			if (string.IsNullOrWhiteSpace(tag))
				throw new ArgumentException("tag cant be null or empty");

			if (isGlobal && tag[0] != GlobalDrawingToolManager.GlobalDrawingToolTagPrefix)
				tag = string.Format("{0}{1}", GlobalDrawingToolManager.GlobalDrawingToolTagPrefix, tag);

			T fibBase = DrawingTool.GetByTagOrNew(owner, typeof(T), tag, templateName) as T;
			if (fibBase == null)
				return null;

			DrawingTool.SetDrawingToolCommonValues(fibBase, tag, isAutoScale, owner, isGlobal);

			// dont nuke existing anchor refs 
			ChartAnchor		startAnchor	= DrawingTool.CreateChartAnchor(owner, startBarsAgo, startTime, startY);
			ChartAnchor		endAnchor	= DrawingTool.CreateChartAnchor(owner, endBarsAgo, endTime, endY);

			startAnchor.CopyDataValues(fibBase.StartAnchor);
			endAnchor.CopyDataValues(fibBase.EndAnchor);
			fibBase.SetState(State.Active);
			return fibBase;
		}

	

		/// <summary>
		/// Draws a fibonacci circle.
		/// </summary>
		/// <param name="owner">The hosting NinjaScript object which is calling the draw method</param>
		/// <param name="tag">A user defined unique id used to reference the draw object</param>
		/// <param name="isAutoScale">Determines if the draw object will be included in the y-axis scale</param>
		/// <param name="startTime">The starting time where the draw object will be drawn.</param>
		/// <param name="startY">The starting y value co-ordinate where the draw object will be drawn</param>
		/// <param name="endTime">The end time where the draw object will terminate</param>
		/// <param name="endY">The end y value co-ordinate where the draw object will terminate</param>
		/// <returns></returns>
		public static PT_FibonacciCircle PT_FibonacciCircle(NinjaScriptBase owner, string tag, bool isAutoScale, 
			DateTime startTime, double startY, DateTime endTime, double endY)
		{
			return PT_FibonacciCore<PT_FibonacciCircle>(owner, isAutoScale, tag, int.MinValue, startTime, startY, int.MinValue, endTime, endY, false, null);
		}

		/// <summary>
		/// Draws a fibonacci circle.
		/// </summary>
		/// <param name="owner">The hosting NinjaScript object which is calling the draw method</param>
		/// <param name="tag">A user defined unique id used to reference the draw object</param>
		/// <param name="isAutoScale">Determines if the draw object will be included in the y-axis scale</param>
		/// <param name="startBarsAgo">The starting bar (x axis co-ordinate) where the draw object will be drawn. For example, a value of 10 would paint the draw object 10 bars back.</param>
		/// <param name="startY">The starting y value co-ordinate where the draw object will be drawn</param>
		/// <param name="endBarsAgo">The end bar (x axis co-ordinate) where the draw object will terminate</param>
		/// <param name="endY">The end y value co-ordinate where the draw object will terminate</param>
		/// <returns></returns>
		public static PT_FibonacciCircle PT_FibonacciCircle(NinjaScriptBase owner, string tag, bool isAutoScale, 
			int startBarsAgo, double startY, int endBarsAgo, double endY)
		{
			return PT_FibonacciCore<PT_FibonacciCircle>(owner, isAutoScale, tag, startBarsAgo, Core.Globals.MinDate, startY, endBarsAgo, Core.Globals.MinDate, endY, false, null);
		}

		/// <summary>
		/// Draws a fibonacci circle.
		/// </summary>
		/// <param name="owner">The hosting NinjaScript object which is calling the draw method</param>
		/// <param name="tag">A user defined unique id used to reference the draw object</param>
		/// <param name="isAutoScale">Determines if the draw object will be included in the y-axis scale</param>
		/// <param name="startTime">The starting time where the draw object will be drawn.</param>
		/// <param name="startY">The starting y value co-ordinate where the draw object will be drawn</param>
		/// <param name="endTime">The end time where the draw object will terminate</param>
		/// <param name="endY">The end y value co-ordinate where the draw object will terminate</param>
		/// <param name="isGlobal">Determines if the draw object will be global across all charts which match the instrument</param>
		/// <param name="templateName">The name of the drawing tool template the object will use to determine various visual properties</param>
		/// <returns></returns>
		public static PT_FibonacciCircle PT_FibonacciCircle(NinjaScriptBase owner, string tag, bool isAutoScale, 
			DateTime startTime, double startY, DateTime endTime, double endY, bool isGlobal, string templateName)
		{
			return PT_FibonacciCore<PT_FibonacciCircle>(owner, isAutoScale, tag, int.MinValue, startTime, startY, int.MinValue, endTime, endY, isGlobal, templateName);
		}

		/// <summary>
		/// Draws a fibonacci circle.
		/// </summary>
		/// <param name="owner">The hosting NinjaScript object which is calling the draw method</param>
		/// <param name="tag">A user defined unique id used to reference the draw object</param>
		/// <param name="isAutoScale">Determines if the draw object will be included in the y-axis scale</param>
		/// <param name="startBarsAgo">The starting bar (x axis co-ordinate) where the draw object will be drawn. For example, a value of 10 would paint the draw object 10 bars back.</param>
		/// <param name="startY">The starting y value co-ordinate where the draw object will be drawn</param>
		/// <param name="endBarsAgo">The end bar (x axis co-ordinate) where the draw object will terminate</param>
		/// <param name="endY">The end y value co-ordinate where the draw object will terminate</param>
		/// <param name="isGlobal">Determines if the draw object will be global across all charts which match the instrument</param>
		/// <param name="templateName">The name of the drawing tool template the object will use to determine various visual properties</param>
		/// <returns></returns>
		public static PT_FibonacciCircle PT_FibonacciCircle(NinjaScriptBase owner, string tag, bool isAutoScale, 
			int startBarsAgo, double startY, int endBarsAgo, double endY, bool isGlobal, string templateName)
		{
			return PT_FibonacciCore<PT_FibonacciCircle>(owner, isAutoScale, tag, startBarsAgo, Core.Globals.MinDate, startY, endBarsAgo,
				Core.Globals.MinDate, endY, isGlobal, templateName);
		}


		/// <summary>
		/// Draws a fibonacci fan.
		/// </summary>
		/// <param name="owner">The hosting NinjaScript object which is calling the draw method</param>
		/// <param name="tag">A user defined unique id used to reference the draw object</param>
		/// <param name="isAutoScale">Determines if the draw object will be included in the y-axis scale</param>
		/// <param name="startTime">The starting time where the draw object will be drawn.</param>
		/// <param name="startY">The starting y value co-ordinate where the draw object will be drawn</param>
		/// <param name="endTime">The end time where the draw object will terminate</param>
		/// <param name="endY">The end y value co-ordinate where the draw object will terminate</param>
		/// <returns></returns>
		public static PT_FibonacciFan PT_FibonacciFan(NinjaScriptBase owner, string tag, bool isAutoScale, 
			DateTime startTime, double startY, DateTime endTime, double endY)
		{
			return PT_FibonacciCore<PT_FibonacciFan>(owner, isAutoScale, tag, int.MinValue, startTime, startY, int.MinValue,
				endTime, endY, false, null);
		}

		/// <summary>
		/// Draws a fibonacci fan.
		/// </summary>
		/// <param name="owner">The hosting NinjaScript object which is calling the draw method</param>
		/// <param name="tag">A user defined unique id used to reference the draw object</param>
		/// <param name="isAutoScale">Determines if the draw object will be included in the y-axis scale</param>
		/// <param name="startBarsAgo">The starting bar (x axis co-ordinate) where the draw object will be drawn. For example, a value of 10 would paint the draw object 10 bars back.</param>
		/// <param name="startY">The starting y value co-ordinate where the draw object will be drawn</param>
		/// <param name="endBarsAgo">The end bar (x axis co-ordinate) where the draw object will terminate</param>
		/// <param name="endY">The end y value co-ordinate where the draw object will terminate</param>
		/// <returns></returns>
		public static PT_FibonacciFan PT_FibonacciFan(NinjaScriptBase owner, string tag, bool isAutoScale, 
			int startBarsAgo, double startY, int endBarsAgo, double endY)
		{
			return PT_FibonacciCore<PT_FibonacciFan>(owner, isAutoScale, tag, startBarsAgo, Core.Globals.MinDate, startY, endBarsAgo, 
				Core.Globals.MinDate, endY, false, null);
		}

		/// <summary>
		/// Draws a fibonacci fan.
		/// </summary>
		/// <param name="owner">The hosting NinjaScript object which is calling the draw method</param>
		/// <param name="tag">A user defined unique id used to reference the draw object</param>
		/// <param name="isAutoScale">Determines if the draw object will be included in the y-axis scale</param>
		/// <param name="startTime">The starting time where the draw object will be drawn.</param>
		/// <param name="startY">The starting y value co-ordinate where the draw object will be drawn</param>
		/// <param name="endTime">The end time where the draw object will terminate</param>
		/// <param name="endY">The end y value co-ordinate where the draw object will terminate</param>
		/// <param name="isGlobal">Determines if the draw object will be global across all charts which match the instrument</param>
		/// <param name="templateName">The name of the drawing tool template the object will use to determine various visual properties</param>
		/// <returns></returns>
		public static PT_FibonacciFan PT_FibonacciFan(NinjaScriptBase owner, string tag, bool isAutoScale, 
			DateTime startTime, double startY, DateTime endTime, double endY, bool isGlobal, string templateName)
		{
			return PT_FibonacciCore<PT_FibonacciFan>(owner, isAutoScale, tag, int.MinValue, startTime, startY, int.MinValue, 
				endTime, endY, isGlobal, templateName);
		}

		/// <summary>
		/// Draws a fibonacci fan.
		/// </summary>
		/// <param name="owner">The hosting NinjaScript object which is calling the draw method</param>
		/// <param name="tag">A user defined unique id used to reference the draw object</param>
		/// <param name="isAutoScale">Determines if the draw object will be included in the y-axis scale</param>
		/// <param name="startBarsAgo">The starting bar (x axis co-ordinate) where the draw object will be drawn. For example, a value of 10 would paint the draw object 10 bars back.</param>
		/// <param name="startY">The starting y value co-ordinate where the draw object will be drawn</param>
		/// <param name="endBarsAgo">The end bar (x axis co-ordinate) where the draw object will terminate</param>
		/// <param name="endY">The end y value co-ordinate where the draw object will terminate</param>
		/// <param name="isGlobal">Determines if the draw object will be global across all charts which match the instrument</param>
		/// <param name="templateName">The name of the drawing tool template the object will use to determine various visual properties</param>
		/// <returns></returns>
		public static PT_FibonacciFan PT_FibonacciFan(NinjaScriptBase owner, string tag, bool isAutoScale, 
			int startBarsAgo, double startY, int endBarsAgo, double endY, bool isGlobal, string templateName)
		{
			return PT_FibonacciCore<PT_FibonacciFan>(owner, isAutoScale, tag, startBarsAgo, Core.Globals.MinDate, startY, endBarsAgo, 
				Core.Globals.MinDate, endY, isGlobal, templateName);
		}
	}
*/
}
